<?php

namespace App\Http\Controllers;

use App\Enum\StatusType;
use App\Enum\VerifyStatus;
use App\Models\ApplyForm;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class VerificationController extends Controller
{
    public function enterOTP()
    {
        return view('verification');
    }

    public function resendOtp()
    {
        $appId = session('app_id');
        $application = ApplyForm::where('id', $appId)->first();
        $otp = random_int(1111, 9999);

        $application->update([
            'otp' => $otp,
            'take_count' => $application->take_count + 1,
        ]);

        if ($application->take_count > 3) {
            return redirect()->back()->with('error', 'You have exceeded the maximum OTP request limit. Please try again later.');
        } else {
            $message = 'Your Otp Code is ' . $application->otp;
        }

        send_sms($application->phone, $message);
        return redirect()->back()->with('success', 'OTP has been sent successfully!');

    }

    public function verifyOTP(Request $request)
    {
        try {
            // Validate OTP input
            $validation = $this->validate($request, [
                'otp' => 'required|numeric',
            ]);

            // Retrieve application by ID
            $appId = session('app_id');
            $application = ApplyForm::where('id', $appId)->first();

            // Check if application exists
            if (!$application) {
                return redirect()->route('home')->with('error', 'Application not found.');
            }

            if ($application->fail_count > 3) {
                $application->update([
                    'status' => 2,
                ]);

                return back()->withErrors([
                    'otp' => 'Too many invalid OTP attempts. Your application is blocked.',
                ]);
            }

            if ($application->otp === $request->otp) {
                $application->fail_count = 0;
                $application->take_count += 1;
                $application->status = StatusType::PENDING;
                $application->otp_verified = VerifyStatus::VERIFIED;
                $application->save();

                return redirect()->route('application-step-one')->with('message', 'OTP verified successfully!');
            } else {
                $application->fail_count += 1;
                $application->save();

                return redirect()->back()->with('error', 'Invalid OTP. Please try again.');
            }

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred. Please try again.');
        }
    }

    public function sendOTP(Request $request)
    {
        $validated = $request->validate([
            'applicant_phone' => 'required',
            'applicant_name' => 'required|string|max:255',
            'accident_date_time' => 'required',
        ]);

        try {
            $next_month = Carbon::parse($validated['accident_date_time'])->addMonth()->format('Y-m-d H:i:s');

            if (Carbon::now() > $next_month) {
                return redirect()->back()->with('error', 'Accident day has passed 30 Days');
            }

            if (Carbon::parse($validated['accident_date_time'])->greaterThan(Carbon::now())) {
                return redirect()->back()->with('error', 'Invalid Date!');
            }

            $otp = random_int(1111, 9999);
            $referenceNumber = generateReferenceNumber();
            $validated['otp'] = $otp;
            $validated['reference_number'] = $referenceNumber;
            $validated['phone'] = $validated['applicant_phone'];

            $application = ApplyForm::create($validated);

            Session::put('app_id', $application->id);

            $message = 'Your Otp Code is ' . $otp;
            $response = send_sms($validated['applicant_phone'], $message);

            if ($response) {
                return redirect()->route('enter-otp');
            } else {
                return back()->with('error', $response);
            }

        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }
}
