<?php

namespace App\Http\Controllers\User;

use App\Enum\EnoType;
use App\Enum\ImageUpload;
use App\Enum\UserType;
use App\Http\Controllers\Controller;
use App\Mail\ApplicationMail;
use App\Models\ApplyForm;
use App\Models\District;
use App\Models\Division;
use App\Models\Upozila;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Session;
use Mccarlosen\LaravelMpdf\Facades\LaravelMpdf as PDF;

class ApplicationController extends Controller
{
    public function index()
    {
        $appId = Session::get('app_id');
        $application = ApplyForm::where('id', $appId)->first();
        $division = Division::select('id', 'bn_name')->get();
        if (!$application || $application->otp_verified != 'verified') {
            return redirect()->route('home')->with('error', 'Application not found.');
        }
        return view('user.applications.index', compact('division'));
    }

    public function todayApplications()
    {
        $user = Auth::user();

        $query = ApplyForm::query()->whereDate('created_at', now()->toDateString())->latest();

        $application = applicationData($query, $user);

        return view('backEnd.application.today-application', compact('application'));
    }

    public function lastMonthApplications()
    {
        $user = Auth::user();

        $query = ApplyForm::query()->whereMonth('created_at', Carbon::now()->subMonth())->latest();

        $application = applicationData($query, $user);

        return view('backEnd.application.today-application', compact('application'));
    }

    public function lastWeekApplications()
    {
        $user = Auth::user();

        $query = ApplyForm::query()->with('district')
            ->whereBetween('created_at', [
                Carbon::now()->subWeek()->startOfWeek(), // Start of last week
                Carbon::now()->subWeek()->endOfWeek(),
            ])->latest();

        $application = applicationData($query, $user);

        return view('backEnd.application.today-application', compact('application'));
    }

    public function applyNow()
    {
        return view('user.applications.apply_now');
    }

    public function create()
    {
        return view('user.applications.index');
    }

    public function update(Request $request, $refId)
    {
        $validatedData = $request->validate([
            'medical_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'death_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'victim_driving_license' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'warish_sonod' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'victim_birth_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'applicant_nid' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'applicant_birth_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'victim_nid' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'non_judicial_stamp' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'applicant_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'gd_stamp' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'inheritance_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'power_of_attorney' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
        ]);

        $application = ApplyForm::where('reference_number', $refId)->first();
        $data = $request->except('_token');

        // Upload Files and Save Paths
        if ($request->hasFile('medical_certificate')) {
            $image = $request->file('medical_certificate');
            $fileName = ImageUpload::fileUpload($image);
            $data['medical_certificate'] = $fileName;
        }

        // Upload Files and Save Paths
        if ($request->hasFile('inheritance_certificate')) {
            $image = $request->file('inheritance_certificate');
            $fileName = ImageUpload::fileUpload($image);
            $data['inheritance_certificate'] = $fileName;
        }

        // Upload Files and Save Paths
        if ($request->hasFile('power_of_attorney')) {
            $image = $request->file('power_of_attorney');
            $fileName = ImageUpload::fileUpload($image);
            $data['power_of_attorney'] = $fileName;
        }

        if ($request->hasFile('death_certificate')) {
            $image = $request->file('death_certificate');
            $fileName = ImageUpload::fileUpload($image);
            $data['death_certificate'] = $fileName;
        }

        if ($request->hasFile('victim_driving_license')) {
            $image = $request->file('victim_driving_license');
            $fileName = ImageUpload::fileUpload($image);
            $data['victim_driving_license'] = $fileName;
        }

        if ($request->hasFile('gd_stamp')) {
            $image = $request->file('gd_stamp');
            $fileName = ImageUpload::fileUpload($image);
            $data['gd_stamp'] = $fileName;
        }

        if ($request->hasFile('victim_birth_certificate')) {
            $image = $request->file('victim_birth_certificate');
            $fileName = ImageUpload::fileUpload($image);
            $data['victim_birth_certificate'] = $fileName;
        }

        if ($request->hasFile('applicant_nid')) {
            $image = $request->file('applicant_nid');
            $fileName = ImageUpload::fileUpload($image);
            $data['applicant_nid'] = $fileName;
        }

        if ($request->hasFile('applicant_signature')) {
            $image = $request->file('applicant_signature');
            $fileName = ImageUpload::fileUpload($image);
            $data['applicant_signature'] = $fileName;
        }

        $application->update($data);

        return redirect()->back();
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'victim_name' => 'required|string|max:255',
            'relation' => 'required|string',
            'father_husband_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'applicant_name' => 'nullable|string|max:255',
            'relation_with_deceased' => 'nullable|string|max:255',
            'victim_birthdate' => 'nullable|date',
            'national_id' => 'required|string|max:20',
            'driving_license' => 'nullable|string|max:255',
            'division_id' => 'required|exists:divisions,id',
            'district_id' => 'required|exists:districts,id',
            'upozila_id' => 'required|exists:upozilas,id',
            'address' => 'nullable|string|max:500',
            'accident_description' => 'required|string',
            // 'accident_date_time' => 'required|date',
            'case_dd_number' => 'nullable|string|max:50',
            'death_date' => 'nullable|date',
            'sick_short_details' => 'nullable|string|max:250',
            'total_expense' => 'nullable|numeric',
            'current_address' => 'nullable|string|max:500',
            'fund_aid' => 'nullable|string|max:255',
            'medical_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'death_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'victim_driving_license' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'warish_sonod' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'victim_birth_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'applicant_nid' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'applicant_birth_certificate' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'victim_nid' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'non_judicial_stamp' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'applicant_signature' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
            'gd_stamp' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
        ]);

        DB::beginTransaction();

        try {
            $appId = Session::get('app_id');

            $application = ApplyForm::where('id', $appId)->first();
            $data = $request->except('_token', 'phone');

            // Upload Files and Save Paths
            if ($request->hasFile('medical_certificate')) {
                $image = $request->file('medical_certificate');
                $fileName = ImageUpload::fileUpload($image);
                $data['medical_certificate'] = $fileName;
            }

            if ($request->hasFile('death_certificate')) {
                $image = $request->file('death_certificate');
                $fileName = ImageUpload::fileUpload($image);
                $data['death_certificate'] = $fileName;
            }

            if ($request->hasFile('victim_driving_license')) {
                $image = $request->file('victim_driving_license');
                $fileName = ImageUpload::fileUpload($image);
                $data['victim_driving_license'] = $fileName;
            }

            if ($request->hasFile('gd_stamp')) {
                $image = $request->file('gd_stamp');
                $fileName = ImageUpload::fileUpload($image);
                $data['gd_stamp'] = $fileName;
            }

            if ($request->hasFile('victim_birth_certificate')) {
                $image = $request->file('victim_birth_certificate');
                $fileName = ImageUpload::fileUpload($image);
                $data['victim_birth_certificate'] = $fileName;
            }

            if ($request->hasFile('applicant_nid')) {
                $image = $request->file('applicant_nid');
                $fileName = ImageUpload::fileUpload($image);
                $data['applicant_nid'] = $fileName;
            }

            if ($request->hasFile('applicant_signature')) {
                $image = $request->file('applicant_signature');
                $fileName = ImageUpload::fileUpload($image);
                $data['applicant_signature'] = $fileName;
            }

            $newUser = User::create([
                'division_id' => $application->division_id,
                'district_id' => $application->district_id,
                'upazila_id' => $application->upozila_id,
                'name' => $application->applicant_name,
                'role_id' => UserType::APPLY,
                'phone' => $application->phone,
                'password' => Hash::make($application->reference_number),
            ]);

            $data['user_id'] = $newUser->id;

            $application->update($data);

            $upozilaUnos = Upozila::where('id', $application->upozila_id)->first();

            Mail::to('shamim.seoexpate@gmail.com')->send(new ApplicationMail($application));

            $divisionUser = User::where('division_id', $application->division_id)->where('uno_type', 'divi')->first();

            // Mail::to($divisionUser->email)->send(new ApplicationMail($application));
            Mail::to('sohel.seoexpatebd@gmail.com')->send(new ApplicationMail($application));

            // $districtUser = User::where('district_id', $application->district_id)->get();

            // foreach ($districtUser as $uno) {
            //     Mail::to($uno)->send(new ApplicationMail($application));
            //     // ApplyMailJob::dispatch($application, $uno->email);
            // }

            DB::commit();
            $message = 'Your application tracking number ' . $application->reference_number;

            $response = send_sms($application->phone, $message);

            return redirect()->route('download-application-pdf');
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('home')->withError($e->getMessage());
        }
    }

    public function downloadApplicationPdf()
    {
        // Retrieve application ID from cache
        $appId = Session::get('app_id');
        // Fetch the application from the database
        $application = ApplyForm::where('id', $appId)->first();

        // If the application exists
        if ($application) {
            // Prepare data for the view
            $data = ['invoice' => $application];

            // Generate the PDF using the 'pdf-view' blade template
            $pdf = PDF::loadView('pdf-view', $data);

            // Define the file path and save the PDF
            $filePath = storage_path('app/public/' . $application->reference_number . '.pdf');
            $pdf->save($filePath);
            Session::forget('app_id');

            // Return the thank-you view with the filename for the PDF download link
            return view('thankyou', ['filename' => $application->reference_number . '.pdf']);
        }

        // If no application found, return 404
        abort(404);
    }

    public function servePdf($filename)
    {
        $filePath = storage_path('app/public/' . $filename);
        if (file_exists($filePath)) {
            return response()->download($filePath)->deleteFileAfterSend(true);
        }

        return response()->json(['error' => 'File not found'], 404);
    }

    public function getDistricts($id)
    {
        $districts = District::where('division_id', $id)->select('id', 'bn_name')->get();

        return response()->json($districts);
    }

    public function getThana($id)
    {
        $districts = Upozila::where('district_id', $id)->select('id', 'bn_name')->get();

        return response()->json($districts);
    }

    public function applications()
    {
        $user = Auth::user();
        $query = ApplyForm::with('district', 'upozila')->latest();
        $application = [];

        $userData = ApplyForm::where('user_id', $user->id)->get();

        $adminAplication = $query->get();

        if ($user->uno_type == EnoType::DIVISIONUNO) {
            $application = $query->where('division_id', $user->division_id)->get();
        } else if ($user->uno_type == EnoType::UPOZILAUNO) {

            $application = $query->where('upozila_id', $user->upozila_id)->get();

        } else if ($user->uno_type == EnoType::DISTRICT) {
            $application = $query->where('district_id', $user->district_id)->get();
        } else {
            $application = $query->where('user_id', $user->id)->get();
        }

        return view('backEnd.dashboard.all-application', compact('application', 'userData', 'adminAplication'));
    }

    public function show($value)
    {
        $user = Auth::user();

        if ($user->uno_type == EnoType::DIVISIONUNO) {
            $application = ApplyForm::with('district', 'division', 'upozila')->where('division_id', $user->division_id)->where('reference_number', $value)->first();
        } else if ($user->uno_type == EnoType::UPOZILAUNO) {
            $application = ApplyForm::with('district', 'upozila')->where('upozila_id', $user->upozila_id)->where('reference_number', $value)->first();
        } else if ($user->uno_type == EnoType::DISTRICT) {
            $application = ApplyForm::with('district', 'upozila')->where('district_id', $user->district_id)->latest()->first();
        }

        return view('backEnd.application.show', compact('application'));
    }

    public function userApplication($id)
    {
        $userData = ApplyForm::where('reference_number', $id)->first();

        return view('backEnd.application.user-data-show', compact('userData'));
    }

    public function documentAdd($id)
    {
        $userData = ApplyForm::where('reference_number', $id)->first();
        $division = Division::select('id', 'name', 'bn_name')->get();

        return view('backEnd.application.user-data-edit', compact('userData', 'division'));
    }
}
