<?php

namespace App\Http\Controllers\Admin;

use App\Enum\UserType;
use App\Http\Controllers\Controller;
use App\Models\User;
use DB;
use Hash;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;

class UserCreateController extends Controller
{
    public function index()
    {
        try {
            $users = User::whereRoleId(UserType::USER)->latest()->get();
            return view('backend.user.index', compact('users'));
        } catch (\Exception $e) {
            return redirect()->back()->withErrors('Error fetching users.');
        }
    }

    public function create()
    {
        try {
            return view('backend.user.create');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors('Error displaying create form.');
        }
    }

    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            // Validate the input data
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'username' => 'required|string|max:255|unique:users',
                'email' => 'required|string|email|max:255|unique:users',
                'password' => 'required|string|min:8',
            ]);

            // Retrieve the role 'user'
            $role = Role::where('name', 'user')->firstOrFail();

            // Hash the password
            $validatedData['password'] = Hash::make($validatedData['password']);

            // Assign role ID to the validated data
            $validatedData['role_id'] = $role->id;

            // Create the user
            $user = User::create($validatedData);

            // Check if the role has any permissions and assign them to the user
            if ($role->permissions()->exists()) {
                $permissions = $role->permissions()->pluck('id')->toArray();
                $user->permissions()->sync($permissions);
            }

            DB::commit();
            return redirect()->route('admin-user.index')->with('success', 'User created successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors('Error creating user.');
        }
    }

    public function show(string $id)
    {
        try {
            // Implement your show logic here if needed
        } catch (\Exception $e) {
            return redirect()->back()->withErrors('Error displaying user.');
        }
    }

    public function edit(string $id)
    {
        try {
            $user = User::findOrFail($id);
            return view('backend.user.edit', compact('user'));
        } catch (ModelNotFoundException $e) {
            return redirect()->back()->withErrors('User not found.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors('Error displaying edit form.');
        }
    }

    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            // Find the user by ID
            $user = User::findOrFail($id);

            // Validate the input data
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'username' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('users')->ignore($user->id),
                ],
                'email' => [
                    'required',
                    'string',
                    'email',
                    'max:255',
                    Rule::unique('users')->ignore($user->id),
                ],
                'password' => 'nullable|string|min:8',
            ]);

            // Update user data
            $user->name = $validatedData['name'];
            $user->username = $validatedData['username'];
            $user->email = $validatedData['email'];

            // Update password only if a new one is provided
            if (!empty($validatedData['password'])) {
                $user->password = Hash::make($validatedData['password']);
            }

            // Save the updated user
            $user->save();

            DB::commit();
            return redirect()->route('admin-user.index')->with('success', 'User updated successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors('Error updating user.');
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $user = User::findOrFail($id);
            $user->delete();

            DB::commit();
            return redirect()->route('admin-user.index')->with('success', 'User deleted successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors('Error deleting user.');
        }
    }


}
