<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\API\BaseController;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules;
use Illuminate\Validation\Rules\Password;

class AuthController extends BaseController
{
    public function userLogin(Request $request)
    {

        // Validate the input fields
        $validator = Validator::make($request->all(), [
            'phone' => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation errors.', ['error' => $validator->errors()], 422);
        }

        $credentials = [
            'phone' => $request->phone,
            'password' => $request->password,
        ];
        // return $credentials;

        if (Auth::attempt($credentials)) {
            $user = Auth::user();
            $token = $user->createToken('API Token')->plainTextToken;
            $success['token'] = $token;
            $success['name'] =$user->name;

            return $this->sendResponse($success, 'User logged in successfully.');
        }

        return $this->sendError('Unauthorized.', ['error' => 'Invalid username/email or password'], 401);
    }

    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'numeric', 'unique:' . User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        if ($validator->fails()) {
            return $this->sendError('Validation errors.', ['error' => $validator->errors()], 422);
        }

        $user = User::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'role_id' => 2,
            'password' => Hash::make($request->password),
        ]);

        if($user){
            $token = $user->createToken('API Token')->plainTextToken;
            $success['token'] = $token;
            $success['name'] = $user->name;

            return $this->sendResponse($success, 'User registration is successfully.');
        }

    }

    public function logout(Request $request)
    {
        try {
            $request->user()->tokens()->delete();

            $success = [];
            return $this->sendResponse($success, 'User logged out successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Logout failed.', ['error' => $e->getMessage()], 500);
        }
    }

    public function update(Request $request)
    {
        try {
            // Validate the input fields
            $validator = Validator::make($request->all(), [
                'current_password' => ['required'],
                'password' => ['required', Password::defaults(), 'confirmed'],
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation errors', ['error' => $validator->errors()], 422);
            }

            // Get the authenticated user
            $user = $request->user();

            // Verify that the provided current password matches the user's current password
            if (!Hash::check($request->input('current_password'), $user->password)) {
                return $this->sendError('Wrong password.', ['error' => 'The provided current password is incorrect.'], 422);
            }

            // Update the user's password
            $user->password = Hash::make($request->input('password'));
            $user->save();

            $success = [];
            return $this->sendResponse($success, 'User password updated successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Server error.', ['error' => $e->getMessage()], 500);
        }
    }

    public function updateProfile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation errors', ['error' => $validator->errors()], 422);
            }

            $user = $request->user();
            $user->name = $request->name;
            $user->save();
            $success['name'] = $user->name;
            return $this->sendResponse($success, 'User updated successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Unauthorised.', ['error' => $e->getMessage()], 500);
        }
    }

    public function userDetails()
    {
        try {
            if (auth()->check()) {
                $userId = auth()->user()->id;
                $user = User::where('id', $userId)->select('id', 'name', 'username', 'email')->first();
                $success['user_details'] = $user;
                return $this->sendResponse($success, 'User details retrieved successfully.');
            } else {
                return $this->sendError('Unauthorized', ['error' => 'User is not authenticated'], 401);
            }
        } catch (\Exception $e) {
            return $this->sendError('Server error', ['error' => $e->getMessage()], 500);
        }
    }

    public function userDeleteAccount()
    {
        try {
            if (auth()->check()) {
                $user = auth()->user();
                $user->delete();

                $success = [];
                return $this->sendResponse($success, 'User deleted successfully.');
            } else {
                return $this->sendError('Unauthorized', ['error' => 'User not authorized'], 401);
            }
        } catch (\Exception $e) {
            return $this->sendError('Server error', ['error' => $e->getMessage()], 500);
        }
    }

    public function getUserOnlineStatus()
    {
        try {
            return $this->sendResponse($success = [], 'User is online');
        } catch (\Exception $e) {
            return $this->sendError('Server error', ['error' => $e->getMessage()], 500);
        }
    }

    public function backup(Request $request)
    {
        try {

            $user = User::findOrFail(Auth::user()->id);

            $validator = Validator::make($request->all(), [
                'backup' => 'required|mimes:doc,txt,docx,xls,xlsx,pdf,sql ' . $request->file('backup')->getClientOriginalExtension(),

            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation errors', ['error' => $validator->errors()], 422);
            }

            if ($request->hasFile('backup')) {

                $oldBackup = $user->backup;
                if (file_exists($oldBackup)) {
                    unlink($oldBackup);
                }

                $document = $request->file('backup');

                $extension = $document->getClientOriginalExtension();
                $fileName = time() . '.' . $extension;
                $document->move('backup/', $fileName);

                $filePath = 'backup/' . $fileName;
                $user->update(['backup' => $filePath]);
            }
            $success = [];
            return $this->sendResponse($success, 'backup file upload successfully.');

        } catch (\Exception $e) {
            return $this->sendError('Server error', ['error' => $e->getMessage()], 500);
        }
    }

    public function userBackup()
    {
        try {
            $user = User::findOrFail(Auth::user()->id);

            $success = [
                'backupFile' => url('/') . '/' . $user->backup,
            ];

            return $this->sendResponse($success, 'Data backup successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Server error', ['error' => $e->getMessage()], 500);
        }
    }

}
